import os
import shutil
from tqdm import tqdm

# --- 1. 配置信息 ---
# 源数据集的基础路径
SOURCE_PATHS = {
    'coco': '',
    'div2k': '',
    'wikiart': ''
}

# 目标数据集的基础路径
DEST_BASE_PATH = ''

# 每个集合所需的图像数量配置
# 格式: '集合名称': {'数据集前缀': 数量, ...}
CONFIG = {
    'train': {
        'coco': 10000,
        'div2k': 500,
        'wikiart': 5000
    },
    'val': {
        'coco': 5000,
        'div2k': 500,
        'wikiart': 5000
    },
    'test': {
        'coco': 5000,
        'div2k': 500,
        'wikiart': 5000
    }
}

# --- 2. 主逻辑函数 ---
def create_dataset():
    """
    根据配置创建新的数据集
    """
    print("开始创建 watermark 数据集...")

    # 遍历 train, val, test 集合
    for split_name, datasets_config in CONFIG.items():
        dest_split_path = os.path.join(DEST_BASE_PATH, split_name)
        print(f"\n正在处理 '{split_name}' 集合，目标路径: {dest_split_path}")
        os.makedirs(dest_split_path, exist_ok=True)

        # 遍历当前集合中需要的数据集 (e.g., coco, div2k, wikiart)
        for dataset_prefix, num_images in datasets_config.items():
            
            # 获取源文件夹路径
            source_split_path = os.path.join(SOURCE_PATHS[dataset_prefix], split_name)
            
            print(f"  -> 从 '{dataset_prefix}' 数据集复制 {num_images} 张图片...")
            print(f"     源路径: {source_split_path}")

            if not os.path.isdir(source_split_path):
                print(f"     [警告] 源路径不存在，跳过: {source_split_path}")
                continue

            # 使用tqdm创建进度条
            # 循环指定次数，按顺序复制文件
            for i in tqdm(range(num_images), desc=f"    拷贝 {dataset_prefix} ({split_name})"):
                # 构建源文件名 (e.g., 00001234.png)
                original_filename = f"{i:08d}.png"
                source_file_path = os.path.join(source_split_path, original_filename)

                # 构建目标文件名 (e.g., coco_00001234.png)
                new_filename = f"{dataset_prefix}_{original_filename}"
                dest_file_path = os.path.join(dest_split_path, new_filename)
                
                # 检查源文件是否存在，防止因数量不足报错
                if os.path.exists(source_file_path):
                    # 复制文件，shutil.copy2会同时复制元数据
                    shutil.copy2(source_file_path, dest_file_path)
                else:
                    print(f"\n[警告] 源文件未找到，已停止在: {source_file_path}")
                    print(f"'{dataset_prefix}' 的 '{split_name}' 集合可能没有 {num_images} 张图片。")
                    break # 停止当前数据集的复制

    print("\n-------------------------------------")
    print("所有任务完成！watermark 数据集创建成功。")
    print(f"数据集位置: {DEST_BASE_PATH}")
    print("-------------------------------------")


# --- 3. 脚本入口 ---
if __name__ == "__main__":
    # 在运行前，确保tqdm已安装。如果未安装，请取消下一行注释并运行一次。
    # os.system("pip install tqdm") 
    create_dataset()